#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include <gm.h>
#include <libldiag.h>

#include "gmstuff.h"

struct gm_port *Myport;

#define SENDSIZE 4000
#define MAX_ROUTE_LEN 32
#define TEST_BYTES (Test_kbytes*1024)
#define NRECV 16
#define NSEND 4
#define RX_TIMEOUT (20 * 2 * 1024)

#define RAW_SEND_PKT_TYPE 0x00000f71
#define X32_ID_PKT_TYPE 0x700f000a

int Test_kbytes;
int Test_id_pkts;
double BWthresh;
unsigned char *Sendbuf[NSEND];
unsigned char *Recvbuf[NRECV];
int Send_free[NSEND];
unsigned int Txseq;
int Route_len;
int Sending;
int Pend_send;
int Report_echo_pkts = 0;
unsigned char My_mac_addr[6];

struct send_pkt {               /* custom raw data packet */
  unsigned int serial;
  unsigned int index;
  unsigned int seqno;
};

struct x32_id_pkt {             /* xbar32 ID packet */
  unsigned int switch_id;
  unsigned char absolute_port;
  unsigned char quadrant_enabled;
  unsigned short pad0;
  unsigned int port_status;
  unsigned int   pad1;
  unsigned int  crc32;
};


struct raw_pkt {
  unsigned int pkt_type;
  union {
    struct send_pkt data;
    struct x32_id_pkt x32_id;
  } pkt;
};
    
void send_callback (struct gm_port *port, void *buf, gm_status_t rc);
void raw_callback (struct gm_port *port, void *x, gm_status_t rc);

extern int Hport;

void
initialize_gm()
{
  gm_status_t rc;
  int boardno;
  int i;

  boardno = 0;

  /* Initialize GM */
  rc = gm_init ();
  if (rc != GM_SUCCESS)
    {
      gm_perror ("gm_init()", rc);
      exit (1);
    }

  /* Open mapper port so we can do raw stuff */
  rc = _gm_mapper_open (&Myport, boardno, GM_API_VERSION);
  if (rc != GM_SUCCESS)
    {
      gm_perror ("gm_open()", rc);
      exit (1);
    }
  printf ("Successfully opened board %d, mapper port\n", boardno);

  /* allocate buffers for sends */
  for (i=0; i<NSEND; ++i) {
    Sendbuf[i] = gm_dma_malloc(Myport, BUFSIZ+MAX_ROUTE_LEN);
    if (Sendbuf[i] == NULL) {
      perror("send buffer allocation");
      exit(1);
    }
  }

  /* allocate and provide buffers for receives */
  for (i=0; i<NRECV; ++i) {
    Recvbuf[i] = gm_dma_malloc(Myport, SENDSIZE+MAX_ROUTE_LEN);
    if (Recvbuf[i] == NULL) {
      perror("receive buffer allocation");
      exit(1);
    }

    /* prepare buffer to receive data */
    _gm_provide_raw_receive_buffer(Myport, Recvbuf[i]);
  }

  /* Get MAC address of this card */
  rc = gm_get_unique_board_id(Myport, My_mac_addr);
  if (rc != GM_SUCCESS) {
    gm_perror("gm_get_unique_board_id", rc);
    exit(1);
  }
}

void
close_gm()
{
}

char pop[]="|/-\\";
int ip=0;

int
test_this_route(
  unsigned char *route,
  int rlen,
  int report)
{
  int i;
  struct x32_id_pkt *phwd;
  static unsigned int serial_no = 0;
  unsigned int rxseq;
  struct raw_pkt *pp;
  unsigned int seq;
  unsigned int bytes_received;
  int pass=1;
  int consumed;
  gm_u64_t end;
  double bw;
  union gm_recv_event *event;
  gm_u64_t start;
  gm_u64_t now;
  gm_u64_t last_rx_time;
  int dropped_packets;
  int rx_packets;
  int more_to_rx;

  putchar(pop[ip&3]);
  putchar(8);
  ++ip;
  fflush(stdout);

  /* for callback */
  Route_len = rlen;

  if (! Test_id_pkts) { /* increment serial number for this session */
    ++serial_no;
    Txseq = rxseq = seq = 0;
  }

  for (i=0; i<NSEND; ++i) {
    struct raw_pkt pkt;
    pp = &pkt;
  
    bcopy(route, Sendbuf[i], rlen);

    if (Test_id_pkts) {
      pp->pkt_type = X32_ID_PKT_TYPE;
  
      /* zero out the bytes to be replaced by the x32 HW so the CRC32 comes out right */
      bzero((char *)&pp->pkt.x32_id, sizeof(pp->pkt.x32_id));
    } else {
      pp->pkt_type = RAW_SEND_PKT_TYPE;
      pp->pkt.data.serial = serial_no;
      pp->pkt.data.index = i;
      pp->pkt.data.seqno = Txseq++;
    }

    bcopy(pp, (Sendbuf[i]+rlen), sizeof(*pp));
  }

  /* We will keep sending data until we receive TEST_BYTES bytes 
   * we will then report the amount of time elapsed and the number of 
   * lost packets
   */
  dropped_packets = 0;
  rx_packets = 0;
  bytes_received = 0;

  /* we are now in sending mode, used by callback */
  Sending = 1;

  /* start the clock */
  start = now = gm_ticks(Myport);
  last_rx_time = now;
  
  /* launch a few sends */
  for (i=0; i<NSEND; ++i) {
    struct raw_pkt pkt;
    pp = &pkt;

    bcopy(Sendbuf[i]+rlen, pp, sizeof(*pp));

#if GM_API_VERSION > 0x20100
    /* printf("(cp)sending pkt, out host port %d\n", Hport); */
    _gm_raw_send_ex (Myport, Sendbuf[i], SENDSIZE + rlen, rlen,
	  send_callback, Sendbuf[i], Hport);
#else
    /* printf("sending sn:%d, seq:%d\n", pp->pkt.data.serial, pp->pkt.data.seqno); */
    _gm_raw_send_with_callback (Myport, Sendbuf[i], SENDSIZE + rlen, rlen,
	  send_callback, Sendbuf[i]);
#endif
  }
  Pend_send = NSEND;
  

  /* Now, wait for them to come back.  Stop when enough data received, or
   * when time since last receive gets too long
   */

  more_to_rx = (Test_id_pkts) ? (rx_packets < Test_id_pkts) : (bytes_received < TEST_BYTES);

  while (more_to_rx && ((now - last_rx_time) < RX_TIMEOUT)) {

    event = gm_receive(Myport);
    now = gm_ticks(Myport);

    switch (GM_RECV_EVENT_TYPE (event)) {

    case GM_RAW_RECV_EVENT:

      pp = (struct raw_pkt *) gm_ntohp (event->recv.buffer);
      /* printf("Got a packet!, type: %08x\n", pp->pkt_type); */

      switch (pp->pkt_type) {

      case RAW_SEND_PKT_TYPE:
        if (Report_echo_pkts) {
          printf("Rx'd Raw packet %d\n", rx_packets);
        }
        if (pp->pkt.data.serial != serial_no) {
          printf("got serial %d, expecting %d!\n", pp->pkt.data.serial, serial_no);
          exit(1);
        }

        /* Accumulate number of dropped packets */
        dropped_packets += seq - rxseq;
        rxseq = seq+1;
        ++rx_packets;
        bytes_received += SENDSIZE;

        break;

      case X32_ID_PKT_TYPE:
        phwd = &pp->pkt.x32_id;
        if (Report_echo_pkts) {
          printf("Rx'd %d ID pkt. Id: %d, port: %u, quad: %u, port status: %08x, crc: %08x\n",
           rx_packets, phwd->switch_id, phwd->absolute_port, phwd->quadrant_enabled,
           phwd->port_status, phwd->crc32);
        }
        ++rx_packets;

      break;

      default:
        /*printf("Got an unk packet!, type: %08x\n", pp->pkt_type); */
	;
      }
      /* hand buffer back to GM */
      _gm_provide_raw_receive_buffer(Myport, pp);

      /* Accumulate number of dropped packets */
      last_rx_time = now;

      break;

    default:
      /* printf("got an unk event, type: %d\n", GM_RECV_EVENT_TYPE (event)); */
      gm_unknown(Myport, event);
    }
    /* printf("bytes rx'd: %d TB: %d\n", bytes_received, TEST_BYTES); */
    more_to_rx = (Test_id_pkts) ? (rx_packets < Test_id_pkts) : (bytes_received < TEST_BYTES);
  }

  /* done, for better or worse, mark end time */
  end = gm_ticks(Myport);

  /* compute bandwidth */
  bw = bytes_received / 
	((double)(end-start)/(2*1024*1024)) /
	(1024*1024);

  /* printf("bw: %f bytes rx'd: %d\n", bw, bytes_received); */
  if ((BWthresh > 1 && bw < BWthresh) ||
      dropped_packets > 0 ||
      (now - last_rx_time) >= RX_TIMEOUT) {
    if (report) {
      for (i=0; i<rlen; ++i) {
       	printf("%02x ", route[i]);
      }
      puts("");
      printf("Received %d of %d packets (dropped %d?)\n",
                	rx_packets, Txseq, dropped_packets);
      printf("BW = %.2fMB/s\n", bw);
      printf("time since last rx = %d usec\n", (int)(now - last_rx_time)/2);
    }
    pass = 0;
  }

  /* now consume any straggling packets */
  Sending = 0;
  consumed = 0;
  now = gm_ticks(Myport);
  last_rx_time = now;

  more_to_rx = (Test_id_pkts) ? Pend_send
      : !(Pend_send == 0 && (rx_packets == 0 || seq >= Txseq-1));

  while (more_to_rx && ((now - last_rx_time) < 10*2*1024*1024)) {
    event = gm_receive(Myport);
    now = gm_ticks(Myport);

    /* printf("got a straggler!(%d)\n", GM_RECV_EVENT_TYPE (event)); */

    switch (GM_RECV_EVENT_TYPE (event)) {

    case GM_RAW_RECV_EVENT:

      pp = (struct raw_pkt *) gm_ntohp (event->recv.buffer);

      switch (pp->pkt_type) {

      case RAW_SEND_PKT_TYPE:
        seq = pp->pkt.data.seqno;
        /* intentional no 'break' */

      case X32_ID_PKT_TYPE:
        ++consumed;
        last_rx_time = now;
        break;

      default:
        ;
      }
    	_gm_provide_raw_receive_buffer(Myport, pp);
      break;

    default:
      /* printf("Unk event type: %d\n",GM_RECV_EVENT_TYPE (event)); */
      gm_unknown(Myport, event);
    }
    more_to_rx = (Test_id_pkts) ? Pend_send
             : !(Pend_send == 0 && (rx_packets == 0 || seq >= Txseq-1));
  }

  if (Pend_send != 0) {
    /* fprintf(stderr, "seq = %d, Txseq = %d\n", seq, Txseq); */
    fprintf(stderr, "Pending sends = %d!\n", Pend_send);
    exit(1);
  }

  /* printf("consumed %d packets, done now\n\n", consumed); */

  return pass;
}

void
send_callback (struct gm_port *port, void *buf, gm_status_t rc)
{
  struct raw_pkt pkt;
  struct raw_pkt *pp;

  pp = &pkt;
  bcopy(buf+Route_len, pp, sizeof(*pp));

  if (rc != GM_SUCCESS) {
      gm_perror ("send error", rc);
      fprintf (stderr, "buffer = %p\n", buf);
      exit (1);
  } else if (Sending) {
    if (! Test_id_pkts) {

      pp->pkt.data.seqno = Txseq++;

      bcopy(pp, buf+Route_len, sizeof(*pp));
    }

#if GM_API_VERSION > 0x20100
      /* printf("(cp)sending pkt, out host port %d\n", Hport); */
      _gm_raw_send_ex (Myport, buf, SENDSIZE+Route_len, Route_len,
                                  send_callback, buf, Hport);
#else
      /* printf("(cp)sending sn:%d, seq:%d\n", pp->serial, pp->seqno); */
      _gm_raw_send_with_callback (Myport, buf, SENDSIZE+Route_len, Route_len,
    	                        send_callback, buf);
#endif
  } else {
    Pend_send--;
  }
}


void
send_invalid_route(
  unsigned char *route,
  int rlen,
  int iterations,
  int gap)
{
  int i;
  gm_u64_t now;
  union gm_recv_event *event;
  static unsigned char *buf;
  volatile int sent;

  if (buf == NULL) {
    buf = gm_dma_malloc(Myport, 32);
    assert(buf != NULL);
  }
  bcopy(route, buf, rlen);
  buf[rlen] = 0;

  /* show what we're sending */
  printf("Sending %d x ", iterations);
  for (i=0; i<rlen+1; ++i) {
    printf("%02x ", buf[i]);
  }
  puts("");

  /* launch the few sends */
  sent = 0;
  for (i=0; i<iterations; ++i) {

#if GM_API_VERSION > 0x20100
    _gm_raw_send_ex (Myport, buf, rlen+1, rlen, raw_callback,
    	(void *)&sent, Hport);
#else
    _gm_raw_send_with_callback (Myport, buf, rlen+1, rlen, raw_callback,
    	(void *)&sent);
#endif

    now = gm_ticks(Myport);
    while (gm_ticks(Myport) < (now+gap))
	;
  }
  
  /* Now, wait for them to complete. */
  while (sent < iterations) {
    void *pp;

    event = gm_receive(Myport);

    switch (GM_RECV_EVENT_TYPE (event)) {

    /* We may get unexpected raw received if other mappers are running */
    case GM_RAW_RECV_EVENT:
      pp = gm_ntohp (event->recv.buffer);

      /* hand buffer back to GM */
      _gm_provide_raw_receive_buffer(Myport, pp);

      break;

    default:
      gm_unknown(Myport, event);
      break;
    }
  }

  printf("sent %d goofy packets\n", iterations);
}

void
raw_callback (struct gm_port *port, void *x, gm_status_t rc)
{
  int *cnt;

  if (rc != GM_SUCCESS)
    {
      gm_perror ("raw send error", rc);
      exit (1);
    }
  else
    {
     cnt = (int *) x;
      ++*cnt;
    }
}


int
get_gm_host_name(
  unsigned char *mac,
  char *name)
{
#if GM_API_VERSION > 0x200    // GM_API_VERSION_2_0
  gm_status_t rc;
  int node_id;
  char lname[GM_MAX_HOST_NAME_LEN+1];
  rc = gm_unique_id_to_node_id(Myport, mac, &node_id);
  if (rc != GM_SUCCESS) {
/*
    gm_perror("gm_unique_id_to_node_id", rc);
*/
    return -1;
  }
  
  rc = gm_node_id_to_host_name_ex(Myport, 2000, node_id, &lname);
  if (rc != GM_SUCCESS) {
/*
    gm_perror("gm_node_id_to_host_name_ex", rc);
*/
    return -1;
  }

  strcpy(name, lname);
  return 0;
#else
  return -1;
#endif
}
